classdef RotMx
    % classdef RotMx - implements class for cartesian 3 vectors, intended to
    % be used with classes vct3 and Frame.
    %
    % Copyright (C) Russell H. Taylor, Johns Hopkins University, 2013
    %
    % Properties
    %  el(3,3)              rotation matric stored in row order
    %
    % methods
    %
    % Notation
    %   R, R1, R2, ...      RotMx objects
    %   v, v1, v2, ...      vct3 objects
    %   eX, eY, eZ          unit X, Y, Z vectors
    %   q                   quaternion (assume unit)
    %
    % Constructors
    %
    % R = RotMx()           Identity rotation
    % R = RotMx(R1)         Copy contructor
    % R = RotMx(M)          Sets R.el to M (no checking)
    % R = RotMx(v)          RotMx(v.X,v.Y,v.Z)
    % R = RotMx(a,b,c)      RotMx(eX,a)*Rot(eY,b)*RotMx(eZ,c)
    % R = RotMx(v,a)        Rotation by angle a about axis v
    % R = RotMx(q)          Rotation matrix corresponding to quaternion q
    %
    % R = RotMx.rand(a,b,c) R = RotMx(A,B,C) where -a<A<a, -b<B<b, -c<C<c 
    % R = RotMx.randD(a,b,c) Same as above, except a, b, c are in degrees
    % R = RotMx.xyz(a,b,c)  RotMx(eX,a)*Rot(eY,b)*RotMx(eZ,c)
    % R = RotMx.xyzD(a,b,c) Same as above, except a, b, c are in degrees                      
    % R = RotMx.eye()       Identity 
    %
    % Computational methods
    %
    % R = mtimes(R1,R2)     Computes R=RotMx(R1.el*R2.el)
    % v = mtimes(R1,v2)     Computes v=vct3(R1.el*v2.el)
    % R = ctranspose(R1)    Computes R=RotMx(R1.el')
    %
    % R = Inverse(R1)       Computes inverse as R1'
    %
    % Methods for dealing with non-orthogonal matrices
    %
    % R = Normalize(R1)     R = VU' from R1=USV'
    % R.NormalizeSelf()     R = vU' from R=USV'
    %
    % Other methods
    %   display(R)          Console display (calls disp)
    %   disp(R)             Basic console display
    % 
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor
    
    properties
        el = [1 0 0; 0 1 0; 0 0 1]
    end
    
    methods
        function R = RotMx(a,b,c)
            switch nargin
                case 0
                    % R.el = [[1 0 0]; [0 1 0]; [0 0 1]];
                    return;
                case 1
                    switch class(a)
                        case 'RotMx'
                            R.el = a.el;
                        case 'vct3'
                            R = RotMx(a.el');
                        case 'quaternion'
                            D=a.norm();
                            if D==0 
                                error('BadArg','Zero quaternion cannot make rotation',a);
                                R.el = zeros(3); return;
                            end
                            x = a.x/D; y=a.y/D; z=a.z/D; r=a.r/D;
                            R.el = [[1-2*(y*y+z*z) 2*(x*y-z*r)   2*(x*z+y*r)]; ...
                                    [2*(x*y+z*r)   1-2*(x*x+z*z) 2*(y*z-x*r)]; ...
                                    [2*(x*z-y*r)   2*(y*z+x*r)   1-2*(x*x+y*y)]];
                        otherwise
                            sz = size(a);
                            if sz==[3 3] % assume3x3 matrix
                                R.el = a;
                            elseif sz==[1 3] % assume R(x,alpha)*R(y,beta)*r(z,gamma)
                                sa = sin(a(1)); ca = cos(a(1)); 
                                sb = sin(a(2)); cb = cos(a(2));
                                sc = sin(a(3)); cc = cos(a(3));
                                R.el = [[ cb*cc          -cb*sc           sb   ]; ...
                                        [ sa*sb*cc+ca*sc -sa*sb*sc+ca*cc -sa*cb]; ...
                                        [-ca*sb*cc+sa*sc  ca*sb*sc+sa*cc  ca*cb]];
                                return;
                            else
                                error('BadArg','Bad Argument to RotMx',a);                      
                                % this should be an error case for now
                            end
                        end
                case 2
                    u = a.unit().el; % axis
                    a = b;        % angle
                    c = cos(a);
                    R.el = diag([c,c,c])+sin(a)*skew(u)+(1-c)*(u*u');      
                case 3
                     sa = sin(a); ca = cos(a); 
                     sb = sin(b); cb = cos(b);
                     sc = sin(c); cc = cos(c);
                     R.el = [[ cb*cc          -cb*sc           sb   ]; ...
                             [ sa*sb*cc+ca*sc -sa*sb*sc+ca*cc -sa*cb]; ...
                             [-ca*sb*cc+sa*sc  ca*sb*sc+sa*cc  ca*cb]];
                     return;
                otherwise
                    error('BadArg','Too many arguments to RotMx',nargin);
            end
                    
        end
        
        function R3 = mtimes(R1, R2)
            switch class(R2)
                case 'RotMx'
                    R3 = RotMx(R1.el*R2.el);
                case 'vct3'
                    R3 = vct3(R1.el*R2.el);
                case 'vct3Array'
                    R3 = vct3Array(R1.el*R2.el);
                otherwise
                    error('BadArg','Bad Argument to RotMx multiply',R2);
            end
        end
        
        function R2 = ctranspose(R)
            R2 = RotMx(R.el');
        end
        
        function Ri = Inverse(R)
            Ri = RotMx(R.el');
        end

        function R = Normalize(R1)
            [U,S,V] = svd(R1.el);
            R = RotMx(V*U');
        end
        
        function [axis,angle] = AxisAngle(R)
            k = [ R.el(3,2)-R.el(2,3);
                  R.el(1,3)-R.el(3,1);
                  R.el(2,1)-R.el(1,2)];
            t = trace(R.el);
            r = norm(k);
            axis = vct3(k/r);
            angle = atan2(r,t-1);
        end
              
        function disp(X)
            if size(X,1)>1
               n = size(X,1)
               for i=1:n disp(X(i,:));end
            else
                if size(X,2)>1
                    n = size(X,1); 
                    for i=1:n disp(X(:,i));end
                else  
                    disp(X.el);
                end
            end
        end
        
        function display(X)          
            if isequal(get(0,'FormatSpacing'),'compact')
               disp([inputname(1) ' =']);
               disp(X);
            else
               disp(' ')
               disp([inputname(1) ' =']);
               disp(' ');
               disp(X);
            end
        end
            
                           
    end
        
    methods (Static)
        function R = rand(aX,aY,aZ)          
            R = RotMx.xyz(2*aX*rand()-aX,2*aY*rand()-aY,2*aZ*rand()-aZ);
        end
        
        function R = randD(aX,aY,aZ)          
            R = RotMx.xyzD(2*aX*rand()-aX,2*aY*rand()-aY,2*aZ*rand()-aZ);
        end
        
        function R = eye()
            R = RotMx(eye(3));
        end
                
        function R = xyz(aX,aY,aZ)
            sa = sin(aX); ca = cos(aX); 
            sb = sin(aY); cb = cos(aY);
            sc = sin(aZ); cc = cos(aZ);
            R  = RotMx([[ cb*cc          -cb*sc           sb   ]; ...
                        [ sa*sb*cc+ca*sc -sa*sb*sc+ca*cc -sa*cb]; ...
                        [-ca*sb*cc+sa*sc  ca*sb*sc+sa*cc  ca*cb]]);    
        end
        
        function R = xyzD(aX,aY,aZ)
            sa = sind(aX); ca = cosd(aX); 
            sb = sind(aY); cb = cosd(aY);
            sc = sind(aZ); cc = cosd(aZ);
            R  = RotMx([[ cb*cc          -cb*sc           sb   ]; ...
                        [ sa*sb*cc+ca*sc -sa*sb*sc+ca*cc -sa*cb]; ...
                        [-ca*sb*cc+sa*sc  ca*sb*sc+sa*cc  ca*cb]]);
        end
        
 
    end
end

